# -*- coding: utf-8 -*-

# =========================================================
# ==      文档批量转换助手 V3.2 (终极修正版)             ==
# =========================================================
#
#   - 修复: 彻底重构“取消与回滚”逻辑，实现真正的“无痕”回滚
#
# =========================================================

import os
import sys
import platform
import subprocess
import shutil
import threading
import queue
from multiprocessing import Pool, cpu_count
import tkinter as tk
from tkinter import ttk, filedialog, messagebox, scrolledtext

# --- (配置区与后端逻辑不变) ---
APP_TITLE = "文档批量转换助手 V3.2"
PDF_SUCCESS_DIR_NAME = "01_转换成功的PDF"; TXT_SUCCESS_DIR_NAME = "02_转换成功的TXT"
FAILED_ORIGINALS_DIR_NAME = "03_转换失败的原始文件"; SUCCESS_ORIGINALS_DIR_NAME = "04_处理成功的原始文件"
SLIDE_PATTERNS = ['.ppt', '.pptx', '.odp']; TEXT_PATTERNS = ['.doc', '.docx', '.rtf', '.odt']

def get_soffice_path():
    system = platform.system()
    if system == "Darwin": return "/Applications/LibreOffice.app/Contents/MacOS/soffice"
    elif system == "Windows":
        path1 = os.path.join(os.environ.get("ProgramFiles", "C:\\Program Files"), "LibreOffice", "program", "soffice.exe")
        if os.path.exists(path1): return path1
        path2 = os.path.join(os.environ.get("ProgramFiles(x86)", "C:\\Program Files (x86)"), "LibreOffice", "program", "soffice.exe")
        if os.path.exists(path2): return path2
    return None

def convert_single_file(args):
    file_path, output_dir, target_format, soffice_path = args
    try:
        cmd = [soffice_path, "--headless", "--convert-to", target_format, "--outdir", output_dir, file_path]
        startupinfo = None
        if platform.system() == "Windows":
            startupinfo = subprocess.STARTUPINFO(); startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW
        subprocess.run(cmd, check=True, capture_output=True, text=True, startupinfo=startupinfo)
        return (file_path, True)
    except Exception:
        return (file_path, False)

# --- GUI 应用层 (包含最终优化) ---
class DocConverterApp:
    def __init__(self, root):
        self.root = root; self.root.title(APP_TITLE); self.root.geometry("600x720"); self.root.resizable(False, False)
        self.soffice_path = get_soffice_path(); self.source_files = []; self.progress_queue = queue.Queue()
        self.cancel_event = threading.Event()
        self.apply_styles(); self.setup_ui(); self.root.protocol("WM_DELETE_WINDOW", self.on_closing); self.check_prerequisites()

    def on_closing(self):
        if hasattr(self, 'worker_thread') and self.worker_thread.is_alive():
            if messagebox.askyesno("确认退出", "转换任务正在进行中，确定要强制退出吗？\n（已转换的文件将不会被保存）"):
                self.cancel_event.set()
                self.root.destroy()
        else:
            self.root.destroy()

    def apply_styles(self):
        self.BG_COLOR = "#F7F9FC"; self.ACCENT_COLOR = "#5B98D1"; self.PROGRESS_BAR_COLOR = "#6CB1E2"; self.TEXT_COLOR = "#333333"; self.SUCCESS_COLOR = "#2E8B57"; self.FAIL_COLOR = "#C0392B"
        style = ttk.Style(self.root); style.theme_use('clam')
        style.configure('.', background=self.BG_COLOR, foreground=self.TEXT_COLOR, font=('Helvetica Neue', 12))
        style.configure('TFrame', background=self.BG_COLOR); style.configure('TLabel', background=self.BG_COLOR, foreground=self.TEXT_COLOR)
        style.configure('TLabelFrame', background=self.BG_COLOR); style.configure('TLabelFrame.Label', background=self.BG_COLOR, foreground=self.TEXT_COLOR, font=('Helvetica Neue', 13, 'bold'))
        style.configure('TRadiobutton', background=self.BG_COLOR, foreground=self.TEXT_COLOR)
        style.configure('Accent.TButton', font=('Helvetica Neue', 14, 'bold'), background=self.ACCENT_COLOR, foreground='white')
        style.map('Accent.TButton', background=[('active', '#7BAEDB'), ('disabled', '#B0C4DE')], foreground=[('disabled', '#F0F0F0')])
        style.configure('Success.TButton', font=('Helvetica Neue', 14, 'bold'), background=self.SUCCESS_COLOR, foreground='white')
        style.map('Success.TButton', background=[('active', '#3CB371')])
        style.configure('Close.TButton', font=('Helvetica Neue', 14, 'bold'))
        style.configure('Cancel.TButton', font=('Helvetica Neue', 14, 'bold'), background=self.FAIL_COLOR, foreground='white')
        style.map('Cancel.TButton', background=[('active', '#E74C3C')])
        style.configure('Custom.Horizontal.TProgressbar', troughcolor=self.BG_COLOR, background=self.PROGRESS_BAR_COLOR, bordercolor=self.BG_COLOR)
    
    def setup_ui(self):
        main_frame = ttk.Frame(self.root, padding="15"); main_frame.pack(fill=tk.BOTH, expand=True)
        self.task_frame = ttk.LabelFrame(main_frame, text=" 1. 请先选择转换任务 ", padding="10"); self.task_frame.pack(fill=tk.X, pady=5)
        self.conversion_type = tk.StringVar(value="slides")
        self.radio_pdf = ttk.Radiobutton(self.task_frame, text="转换幻灯片 (PPT, PPTX 等) 为 PDF", variable=self.conversion_type, value="slides")
        self.radio_pdf.pack(anchor=tk.W)
        self.radio_txt = ttk.Radiobutton(self.task_frame, text="转换文档 (DOC, DOCX 等) 为 TXT", variable=self.conversion_type, value="text")
        self.radio_txt.pack(anchor=tk.W)
        self.select_frame = ttk.LabelFrame(main_frame, text=" 2. 再选择文件或文件夹 ", padding="10"); self.select_frame.pack(fill=tk.X, pady=10)
        self.select_button = ttk.Button(self.select_frame, text="📂 选择文件夹...", command=self.select_folder, style='Accent.TButton'); self.select_button.pack(fill=tk.X, ipady=5)
        self.file_list_label = ttk.Label(self.select_frame, text="待处理文件列表:"); self.file_list_label.pack(anchor=tk.W, pady=(10, 2))
        self.file_list_text = scrolledtext.ScrolledText(self.select_frame, height=8, state='disabled', relief=tk.SOLID, borderwidth=1); self.file_list_text.pack(fill=tk.X, expand=True, pady=(10,0))
        self.run_frame = ttk.LabelFrame(main_frame, text=" 3. 开始执行 ", padding="10"); self.run_frame.pack(fill=tk.X, pady=5)
        ttk.Label(self.run_frame, text="总进度:").pack(anchor=tk.W)
        self.total_progress_container = ttk.Frame(self.run_frame); self.total_progress_container.pack(fill=tk.X, pady=2)
        self.total_progress = ttk.Progressbar(self.total_progress_container, style='Custom.Horizontal.TProgressbar', orient='horizontal', mode='determinate'); self.total_progress.pack(fill=tk.X, expand=True, ipady=8)
        self.total_progress_label = ttk.Label(self.run_frame, text="", anchor=tk.CENTER); self.total_progress_label.place(in_=self.total_progress, relx=0.5, rely=0.5, anchor='center')
        self.current_file_label = ttk.Label(self.run_frame, text="当前文件: (等待中)"); self.current_file_label.pack(anchor=tk.W, pady=(10, 2))
        self.current_file_progress = ttk.Progressbar(self.run_frame, style='Custom.Horizontal.TProgressbar', orient='horizontal', mode='indeterminate'); self.current_file_progress.pack(fill=tk.X, pady=2, ipady=2)
        self.status_label = ttk.Label(self.run_frame, text="状态: 一切就绪", wraplength=550); self.status_label.pack(anchor=tk.W, pady=(10, 10))
        self.button_container = ttk.Frame(self.run_frame); self.button_container.pack(fill=tk.X, pady=10)
        self.start_button = ttk.Button(self.button_container, text="⚡️ 开始转换", command=self.start_conversion, style='Accent.TButton', state='disabled'); self.start_button.pack(fill=tk.X, ipady=8)

    def set_controls_state(self, state):
        self.select_button.config(state=state); self.radio_pdf.config(state=state); self.radio_txt.config(state=state)

    def show_completion_buttons(self):
        for widget in self.button_container.winfo_children(): widget.destroy()
        self.restart_button = ttk.Button(self.button_container, text="✅ 再来一次", command=self.select_folder, style='Success.TButton'); self.restart_button.pack(side=tk.LEFT, expand=True, fill=tk.X, ipady=8, padx=(0, 5))
        self.close_button = ttk.Button(self.button_container, text="✖️ 关闭程序", command=self.on_closing, style='Close.TButton'); self.close_button.pack(side=tk.RIGHT, expand=True, fill=tk.X, ipady=8, padx=(5, 0))

    def reset_to_start_button(self):
        for widget in self.button_container.winfo_children(): widget.destroy()
        self.start_button = ttk.Button(self.button_container, text="⚡️ 开始转换", command=self.start_conversion, style='Accent.TButton', state='disabled'); self.start_button.pack(fill=tk.X, ipady=8)

    def update_progress_text(self, value, maximum):
        if maximum > 0: percent = int((value / maximum) * 100); self.total_progress_label.config(text=f"{percent}% ({value}/{maximum})")
        else: self.total_progress_label.config(text="")
            
    def check_prerequisites(self):
        if not (self.soffice_path and os.path.exists(self.soffice_path)):
            messagebox.showerror("依赖缺失", "错误: 未在默认路径找到 LibreOffice！\n\n请先访问 https://www.libreoffice.org 下载并安装，然后重新启动本应用。"); self.root.quit()

    def select_folder(self):
        self.set_controls_state('normal'); self.reset_to_start_button(); self.file_list_label.config(text="待处理文件列表:"); self.status_label.config(text="状态: 一切就绪")
        self.current_file_label.config(text="当前文件: (等待中)"); self.total_progress.config(value=0); self.update_progress_text(0,0)
        target_dir = filedialog.askdirectory(title="请选择一个包含待处理文件的文件夹");
        if not target_dir: return
        self.target_dir = target_dir; self.file_list_text.config(state='normal'); self.file_list_text.delete('1.0', tk.END); self.source_files.clear()
        patterns = SLIDE_PATTERNS if self.conversion_type.get() == "slides" else TEXT_PATTERNS
        for item in os.listdir(target_dir):
            if os.path.isfile(os.path.join(target_dir, item)) and os.path.splitext(item)[1].lower() in patterns:
                self.source_files.append(os.path.join(target_dir, item)); self.file_list_text.insert(tk.END, os.path.basename(item) + '\n')
        self.file_list_text.config(state='disabled'); self.status_label.config(text=f"状态: 已扫描到 {len(self.source_files)} 个待处理文件。")
        self.start_button.config(state='normal' if self.source_files else 'disabled')

    def start_conversion(self):
        if not self.source_files: messagebox.showwarning("无文件", "没有找到可以处理的文件。"); return
        self.set_controls_state('disabled'); self.cancel_event.clear()
        for widget in self.button_container.winfo_children(): widget.destroy()
        self.cancel_button = ttk.Button(self.button_container, text="🚫 中途取消", command=self.cancel_conversion, style='Cancel.TButton'); self.cancel_button.pack(fill=tk.X, ipady=8)
        self.worker_thread = threading.Thread(target=self.conversion_worker, daemon=True); self.worker_thread.start(); self.process_queue()
    
    def cancel_conversion(self):
        self.status_label.config(text="状态: 正在发送取消信号，请稍候...")
        self.cancel_event.set(); self.cancel_button.config(state='disabled')

    def process_queue(self):
        try:
            msg = self.progress_queue.get_nowait()
            if 'total_max' in msg: self.total_progress.config(maximum=msg['total_max'])
            if 'total_value' in msg: self.total_progress.config(value=msg['total_value']); self.update_progress_text(msg['total_value'], self.total_progress['maximum'])
            if 'current_file_running' in msg:
                if msg['current_file_running']: self.current_file_progress.start(15)
                else: self.current_file_progress.stop()
            if 'current_file_text' in msg: self.current_file_label.config(text=f"当前文件: {msg['current_file_text']}")
            if 'status_text' in msg: self.status_label.config(text=f"状态: {msg['status_text']}")
            if 'final_failed_list' in msg:
                self.file_list_label.config(text=f"最终失败文件清单 ({len(msg['final_failed_list'])}个):"); self.file_list_text.config(state='normal'); self.file_list_text.delete('1.0', tk.END)
                if msg['final_failed_list']:
                    self.file_list_text.tag_configure("fail_style", foreground=self.FAIL_COLOR)
                    for item in msg['final_failed_list']: self.file_list_text.insert(tk.END, os.path.basename(item) + '\n', "fail_style")
                else:
                    self.file_list_text.tag_configure("success_style", foreground=self.SUCCESS_COLOR)
                    self.file_list_text.insert(tk.END, "无失败文件，全部成功！\n", "success_style")
                self.file_list_text.config(state='disabled')
            if 'task_done' in msg:
                self.set_controls_state('normal'); self.show_completion_buttons(); self.total_progress_label.config(text="完成!", foreground=self.SUCCESS_COLOR)
                if msg.get('open_folder_path') and msg.get('task_cancelled') == False:
                    if messagebox.askyesno("任务完成", "所有文件已处理完毕！\n\n是否立即打开输出文件夹？"): self.open_folder(msg['open_folder_path'])
                elif msg.get('task_cancelled') == True:
                    messagebox.showinfo("任务取消", "任务已按您的指示取消。")
                return
        except queue.Empty: pass
        self.root.after(100, self.process_queue)
    
    def open_folder(self, path):
        if platform.system() == "Windows": os.startfile(path)
        elif platform.system() == "Darwin": subprocess.Popen(["open", path])
        else: subprocess.Popen(["xdg-open", path])

    def conversion_worker(self):
        conversion_type = self.conversion_type.get()
        if conversion_type == "slides": target_format, output_dir_name = ("pdf", PDF_SUCCESS_DIR_NAME)
        else: target_format, output_dir_name = ("txt", TXT_SUCCESS_DIR_NAME)
        success_output_dir = os.path.join(self.target_dir, output_dir_name); failed_originals_dir = os.path.join(self.target_dir, FAILED_ORIGINALS_DIR_NAME); success_originals_dir = os.path.join(self.target_dir, SUCCESS_ORIGINALS_DIR_NAME)
        all_output_dirs = [success_output_dir, failed_originals_dir, success_originals_dir]
        for d in all_output_dirs: os.makedirs(d, exist_ok=True)
        files_to_process_master = list(self.source_files); total_files = len(files_to_process_master)
        num_processes = max(1, cpu_count() - 1)
        self.progress_queue.put({'total_max': total_files, 'total_value': 0}); self.update_progress_text(0, total_files)
        successful_paths = set(); files_to_retry_queue = list(files_to_process_master)

        for i in range(3):
            if not files_to_retry_queue or self.cancel_event.is_set(): break
            pass_num = i + 1; current_pass_total = len(files_to_retry_queue)
            self.progress_queue.put({'current_file_running': True})
            tasks = [(f, success_output_dir, target_format, self.soffice_path) for f in files_to_retry_queue]
            
            with Pool(processes=num_processes) as pool:
                results = pool.imap_unordered(convert_single_file, tasks)
                pass_processed_count = 0; pass_fail_count = 0
                for file_path, success in results:
                    if self.cancel_event.is_set(): pool.terminate(); break
                    pass_processed_count += 1
                    if success: successful_paths.add(file_path)
                    else: pass_fail_count += 1
                    status_text = f"第 {pass_num}/3 轮... (总数:{total_files} | 总成功:{len(successful_paths)} | 本轮已处理:{pass_processed_count}/{current_pass_total} | 本轮失败:{pass_fail_count})"
                    self.progress_queue.put({'total_value': len(successful_paths), 'current_file_text': os.path.basename(file_path), 'status_text': status_text})
            
            if self.cancel_event.is_set(): break
            files_to_retry_queue = list(set(files_to_process_master) - successful_paths)
        
        self.progress_queue.put({'status_text': "任务结束，正在整理文件...", 'current_file_running': False})
        
        task_cancelled = self.cancel_event.is_set()
        keep_results = True
        if task_cancelled:
            keep_results = messagebox.askyesno("任务已取消", f"任务已被中途取消。\n\n是否要保留已经成功转换的 {len(successful_paths)} 个文件？")

        final_success_count = 0; final_fail_count = 0; final_failed_list = []
        
        if task_cancelled and not keep_results: # 彻底回滚
            self.progress_queue.put({'status_text': "正在回滚操作，请稍候..."})
            for success_path in successful_paths:
                base, _ = os.path.splitext(os.path.basename(success_path))
                output_file = os.path.join(success_output_dir, base + '.' + target_format)
                if os.path.exists(output_file): os.remove(output_file)
            final_failed_list = [] # 回滚后没有失败清单
            final_fail_count = total_files
        else: # 正常完成 或 取消但保留结果
            successful_paths_set = set(successful_paths)
            for src_path in self.source_files:
                if src_path in successful_paths_set:
                    shutil.move(src_path, os.path.join(success_originals_dir, os.path.basename(src_path)))
                    final_success_count += 1
                else:
                    shutil.move(src_path, os.path.join(failed_originals_dir, os.path.basename(src_path)))
                    final_fail_count += 1
                    final_failed_list.append(src_path)

        for d in all_output_dirs:
            try:
                if os.path.isdir(d) and not os.listdir(d):
                    os.rmdir(d)
            except OSError: pass

        self.progress_queue.put({
            'total_value': final_success_count,
            'status_text': f"全部完成！(总数: {total_files} | 最终成功: {final_success_count} | 最终失败: {final_fail_count})",
            'current_file_text': "已完成",
            'final_failed_list': final_failed_list,
            'task_done': True,
            'task_cancelled': task_cancelled,
            'open_folder_path': self.target_dir
        })

if __name__ == "__main__":
    if platform.system() == "Darwin":
        import multiprocessing
        if multiprocessing.get_start_method(allow_none=True) != 'spawn':
            multiprocessing.set_start_method('spawn', force=True)
    root = tk.Tk()
    app = DocConverterApp(root)
    root.mainloop()